Attribute VB_Name = "PEC"
'---------------------------------------------------------------------
' Copyright  2007 EQMOD Development Team
'
' Permission is hereby granted to use this Software for any purpose
' including combining with commercial products, creating derivative
' works, and redistribution of source or binary code, without
' limitation or consideration. Any redistributed copies of this
' Software must include the above Copyright Notice.
'
' THIS SOFTWARE IS PROVIDED "AS IS". THE AUTHOR OF THIS CODE MAKES NO
' WARRANTIES REGARDING THIS SOFTWARE, EXPRESS OR IMPLIED, AS TO ITS
' SUITABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
'---------------------------------------------------------------------
'
' PEC.bas - Periodic Error Correction functions for EQMOD ASCOM Driver
'
'
' Written:  12-Oct-07   Chris Shillito
'
' Edits:
'
' When      Who     What
' --------- ---     --------------------------------------------------
'---------------------------------------------------------------------
'
'
'  SYNOPSIS:
'
'  This is a demonstration of a EQ6/ATLAS/EQG direct stepper motor control access
'  using the EQCONTRL.DLL driver code.
'
'  File EQCONTROL.bas contains all the function prototypes of all subroutines
'  encoded in the EQCONTRL.dll
'
'  The EQ6CONTRL.DLL simplifies execution of the Mount controller board stepper
'  commands.
'
'  The mount circuitry needs to be modified for this test program to work.
'  Circuit details can be found at http://sourceforge.net/projects/eq-mod/
'

'  DISCLAIMER:

'  You can use the information on this site COMPLETELY AT YOUR OWN RISK.
'  The modification steps and other information on this site is provided
'  to you "AS IS" and WITHOUT WARRANTY OF ANY KIND, express, statutory,
'  implied or otherwise, including without limitation any warranty of
'  merchantability or fitness for any particular or intended purpose.
'  In no event the author will  be liable for any direct, indirect,
'  punitive, special, incidental or consequential damages or loss of any
'  kind whether or not the author  has been advised of the possibility
'  of such loss.

'  WARNING:

'  Circuit modifications implemented on your setup could invalidate
'  any warranty that you may have with your product. Use this
'  information at your own risk. The modifications involve direct
'  access to the stepper motor controls of your mount. Any "mis-control"
'  or "mis-command"  / "invalid parameter" or "garbage" data sent to the
'  mount could accidentally activate the stepper motors and allow it to
'  rotate "freely" damaging any equipment connected to your mount.
'  It is also possible that any garbage or invalid data sent to the mount
'  could cause its firmware to generate mis-steps pulse sequences to the
'  motors causing it to overheat. Make sure that you perform the
'  modifications and testing while there is no physical "load" or
'  dangling wires on your mount. Be sure to disconnect the power once
'  this event happens or if you notice any unusual sound coming from
'  the motor assembly.
'
'  CREDITS:
'
'  Portions of the information on this code should be attributed
'  to Mr. John Archbold from his initial observations and analysis
'  of the interface circuits and of the ASCII data stream between
'  the Hand Controller (HC) and the Go To Controller.
'

Option Explicit

Public PEC_Enabled As Boolean

Type PECData
    Time As Double
    PEPosition As Double
    PECPosition As Double
    RawPosn As Double
    signal As Double
    PErate As Double
    PECrate As Double
    cycle As Integer
End Type

Public PECCurve() As PECData

'Private LastSignal As Double
'Private lastrate As Double
Private CurrRate As Double                  ' current rate
Private PEC_File As String                  ' path and name of PEC file
Private firsttime As Boolean                ' oneshot flag
Private MaxPE As Double                     ' MAX error (for graphing)
Private MinPE As Double                     ' MIX error (for graphing)
Private newpos As Double
Private oldpos As Double
Private timerflag As Boolean                ' timer interlock
Private threshold As Double                 ' minimum correction PEC will make
Private gain As Double                      ' current gain setting
Private PhaseAdjust As Integer              ' current phase adjustment (samples)
Private PhaseShift As Double                ' current phase shift (steps)
Private seccount As Integer                 ' Current PEC sample
Private syncidx As Integer                  ' Sample when resynch occurs
Private gMaxRateAdjust As Double            ' Maximum correction PEC is allowed to make
Private MaxRate As Double                   ' Fastset rate allowed
Private MinRate As Double                   ' slowest rate allowed
Private SID_RATE_NORTH As Double            ' 15.041067        ' arcsecs/sec  (60*60*360) / ((23*60*60)+(56*60)+4)
Private SID_RATE_SOUTH As Double            ' -15.041067       ' arcsecs/sec

Const ARCSECS_PER_360DEGREES = 1296000      ' 360*60*60

Public Sub PEC_PhaseScroll_Change()
Dim adj As Double
    HC.Label45.Caption = CStr(Int(360 * (HC.PhaseScroll.Value / gRAWormPeriod))) & " deg."
    PhaseAdjust = HC.PhaseScroll.Value
    PhaseShift = HC.PhaseScroll.Value * (gRAWormSteps / gRAWormPeriod)
    seccount = GetIdx(seccount)

End Sub

Public Sub PEC_Initialise()
    firsttime = True
    HC.CmdPecSave.Enabled = False
    HC.GainScroll.Enabled = False
    HC.PhaseScroll.Enabled = False
    
    ReDim PECCurve(480)
    SID_RATE_NORTH = gSiderealRate
    SID_RATE_SOUTH = -1 * gSiderealRate
    Call PEC_ReadParams
    
    If gHemisphere Then
        MaxRate = SID_RATE_SOUTH + gMaxRateAdjust
        MinRate = SID_RATE_SOUTH - gMaxRateAdjust
    Else
        MaxRate = SID_RATE_NORTH + gMaxRateAdjust
        MinRate = SID_RATE_NORTH - gMaxRateAdjust
    End If
    
    If (gTot_RA <> 0) Then
        If (gRAWormPeriod > 0) Then
            ReDim PECCurve(gRAWormPeriod)
            HC.PhaseScroll.max = gRAWormPeriod
            Call Import
        End If
    End If
End Sub

Public Sub PEC_Timestamp()
    Dim key As String
    Dim Ini As String
    Dim pos As Double
    Dim temp As String

    Ini = HC.oPersist.GetIniPath & "\EQMOD.ini"
    key = "[pec]"
    gEmulRA = EQ_GetMotorValues(0)
    gLast_time = EQnow_lst_norange()
    gCurrent_time = gLast_time
    gEmulRA_Init = gEmulRA
    pos = NormalisePosition(gEmulRA)
    temp = Now
    Call HC.oPersist.WriteIniValueEx("SYNCPOS", CStr(pos), key, Ini)
    Call HC.oPersist.WriteIniValueEx("SYNCTIME", temp, key, Ini)
End Sub

Public Sub PEC_StartTracking()
    PEC_Enabled = True
    If gHemisphere Then
        Call EQPecMoveAxis(0, (SID_RATE_NORTH) / 3600)
    Else
        Call EQPecMoveAxis(0, (SID_RATE_SOUTH) / 3600)
    End If
    HC.Label28.Caption = FormatNumber(0, 4)
    HC.PECTimer.Enabled = True
End Sub

Public Sub PEC_StopTracking()
    PEC_Enabled = False
    HC.Label28.Caption = FormatNumber(0, 4)
    gRA_LastRate = 0
End Sub

Public Sub PEC_Unload()
    Call PEC_WriteParams
End Sub

Public Sub PEC_GainScroll_Change()
    gain = HC.GainScroll.Value / 10
    HC.Label43.Caption = "x" & CStr(gain)
    If HC.GainScroll.Enabled Then
        Call CalcRates
    End If
End Sub

Public Sub PEC_Load()
    FileDlg.filter = "*.txt*"
    FileDlg.Show (1)
    If FileDlg.filename <> "" Then
        PEC_File = FileDlg.filename
        If Import Then
            PEC_WriteParams
            HC.Change_Display (2)
        End If
    End If
End Sub

Public Sub PEC_Save()
    FileDlg.filter = "*.txt*"
    FileDlg.Show (1)
    If FileDlg.filename <> "" Then
        PEC_File = FileDlg.filename
        Call Export
        HC.PhaseScroll.Value = 0
        Call PEC_WriteParams
        Call Import
        Call PEC_PlotCurve
    End If
End Sub

Public Sub PEC_Timer()
Dim RA As Double
Dim rate As Double
Dim ratechange As Double
Dim x As Integer
    
    On Error Resume Next
    If Not timerflag Then
        timerflag = True
        If gTrackingStatus <> 0 Then
            'only maintain pe trace updates if we're tracking
            
            ' determine PEC index direct from motor position + phase shift
            seccount = GetIdx(seccount)
            
            HC.plot.DrawMode = 7
            newpos = seccount * HC.plot.ScaleWidth / gRAWormPeriod
            If Not firsttime Then
                HC.plot.Line (oldpos, 0)-(oldpos, HC.plot.ScaleHeight), vbRed
            End If
            HC.plot.Line (newpos, 0)-(newpos, HC.plot.ScaleHeight), vbRed
            oldpos = newpos
            firsttime = False
            HC.Label41.Caption = CStr(Int(NormalisePosition(gEmulRA)))
            HC.Label42.Caption = CStr(Int(PECCurve(seccount).PEPosition))
            
            If PEC_Enabled Then
                rate = PECCurve(seccount).PECrate
                If rate <> CurrRate Then
                
                    ' apply the min/max limits - just in case there's
                    ' an error in the rate calculations this prevents'
                    ' the mount from ever slewing wildly!
                    If rate > MaxRate Then
                        rate = MaxRate
                    Else
                        If rate < MinRate Then
                            rate = MinRate
                        End If
                    End If
                    
                    HC.Label28.Caption = FormatNumber(rate - gSiderealRate, 3)
                    Call EQPecMoveAxis(0, (rate / 3600))
                    CurrRate = rate
                End If
            End If
        End If
    End If
    timerflag = False

End Sub
Public Sub PEC_ReadParams()

    Dim tmptxt As String
    Dim i As Integer
    Dim key As String
    Dim Ini As String

    Ini = HC.oPersist.GetIniPath & "\EQMOD.ini"
    key = "[pec]"
     
    tmptxt = HC.oPersist.ReadIniValueEx("THRESHOLD", key, Ini)
    If tmptxt <> "" Then
        threshold = val(tmptxt)
    Else
       ' no value exists - create a default
        threshold = 0#
        Call HC.oPersist.WriteIniValueEx("THRESHOLD", CStr(threshold), key, Ini)
    End If
    
    tmptxt = HC.oPersist.ReadIniValueEx("GAIN", key, Ini)
    If tmptxt <> "" Then
       gain = val(tmptxt)
    Else
       ' no value exists - create a default
       gain = 1.5
       Call HC.oPersist.WriteIniValueEx("GAIN", CStr(gain), key, Ini)
    End If
    HC.GainScroll.Value = gain * 10
    Call PEC_GainScroll_Change
   
    PEC_File = HC.oPersist.ReadIniValueEx("PEC_FILE", key, Ini)
    If PEC_File = "" Then
       PEC_File = "pec.txt"
       ' no value exists - create a default
       Call HC.oPersist.WriteIniValueEx("PEC_FILE", PEC_File, key, Ini)
    End If
    
    tmptxt = HC.oPersist.ReadIniValueEx("PHASE_SHIFT", key, Ini)
    If tmptxt <> "" Then
       HC.PhaseScroll.Value = val(tmptxt)
    Else
       ' no value exists - create a default
       Call HC.oPersist.WriteIniValueEx("PHASE_SHIFT", "0", key, Ini)
       HC.PhaseScroll.Value = 0
    End If
    Call PEC_PhaseScroll_Change
    
    tmptxt = HC.oPersist.ReadIniValueEx("MAX_RATEADJUST", key, Ini)
    If tmptxt <> "" Then
        gMaxRateAdjust = val(tmptxt)
        If gMaxRateAdjust < 3 Then
            ' fix to increse previous default of 1
            gMaxRateAdjust = 3
            Call HC.oPersist.WriteIniValueEx("MAX_RATEDAJUST", "3", key, Ini)
        End If
    Else
       ' no value exists - create a default
       Call HC.oPersist.WriteIniValueEx("MAX_RATEDAJUST", "3", key, Ini)
       gMaxRateAdjust = 3
    End If
End Sub
Public Sub PEC_WriteParams()
    Dim key As String
    Dim Ini As String

    Ini = HC.oPersist.GetIniPath & "\EQMOD.ini"
    key = "[pec]"
    Call HC.oPersist.WriteIniValueEx("THRESHOLD", CStr(threshold), key, Ini)
    Call HC.oPersist.WriteIniValueEx("GAIN", CStr(gain), key, Ini)
    Call HC.oPersist.WriteIniValueEx("PEC_FILE", PEC_File, key, Ini)
    Call HC.oPersist.WriteIniValueEx("PHASE_SHIFT", CStr(HC.PhaseScroll.Value), key, Ini)

End Sub

Private Function NormalisePosition(ByVal Position As Double) As Double
' Normalisation is intended for raw stepper position which are centered
' around H80000. Once normalised the range will be 0-50132.

    If Position > gRAWormSteps Then
        While Position < &H800000
            Position = Position + gTot_RA
        Wend
        NormalisePosition = Position Mod (gRAWormSteps)
    Else
        ' don't take into account the H80000 ofset if the position
        ' is already normalised!
        NormalisePosition = Position
    End If

End Function
Private Function Import() As Boolean
Dim temp1 As String
Dim temp2 As String
Dim lineno As Integer
Dim idx As Integer
Dim pos As Integer
Dim CurveMin As PECData
Dim CurveMax As PECData
Dim ratesum As Double
Dim motorpos As Double
Dim CycleCount As Integer
Dim err As Boolean
Dim drift As Double


    On Error GoTo ImportError
    
    err = False
    temp1 = PEC_File
    Open temp1 For Input As #1
    If err <> 0 Then
      ' File operation(s) failed, handle the error
       HC.Add_Message ("PEC:file open error")
       err = True
    Else
        For idx = 0 To (gRAWormPeriod - 1)
            PECCurve(idx).signal = 0
        Next idx
        lineno = 0
        idx = 0
        While Not EOF(1)
            Line Input #1, temp1
            If lineno > 0 Then
                If Left$(temp1, 1) = "!" Then
                    ' parse parameters
                    pos = InStr(temp1, "=")
                    If pos <> 0 Then
                        temp2 = Left$(temp1, pos - 1)
                        If temp2 = "!WormPeriod" Then
                            temp1 = Right$(temp1, Len(temp1) - pos)
                            If gRAWormPeriod <> Int(val(temp1) + 0.5) Then
                                ' worm period doesn't match our mount!
                                err = True
                                GoTo closefile
                            End If
                        End If
                    End If
                Else
                    If Left$(temp1, 1) <> "#" Then
                        ' replace tabs with spaces
                        temp1 = Replace(temp1, Chr(9), " ")
                        pos = InStr(temp1, " ")
                        If pos <> 0 Then
                            temp2 = Left$(temp1, pos - 1)
                            temp1 = Right$(temp1, Len(temp1) - pos)
                            PECCurve(idx).Time = val(temp2)
                            
                            pos = InStr(temp1, " ")
                            If pos <> 0 Then
                                temp2 = Left$(temp1, pos - 1)
                                temp1 = Right$(temp1, Len(temp1) - pos)
                                If CycleCount = 0 Then
                                    ' store the motor positions for the first cycle
                                    motorpos = val(temp2)
                                    PECCurve(idx).RawPosn = motorpos
                                    PECCurve(idx).PEPosition = NormalisePosition(Int(motorpos))
                                End If
                                PECCurve(idx).signal = (PECCurve(idx).signal + val(temp1))
                                PECCurve(idx).cycle = CycleCount + 1
                            End If
                            idx = idx + 1
                            If idx = gRAWormPeriod Then
                                CycleCount = CycleCount + 1
                                idx = 0
                            End If
                        End If
                    End If
                End If
            End If
            lineno = lineno + 1
        Wend
    End If
closefile:
    Close #1
    If err Then GoTo errCheck
    
    If CycleCount >= 1 Then
        ' average the signal
        For idx = 0 To (gRAWormPeriod - 1)
            PECCurve(idx).signal = PECCurve(idx).signal / PECCurve(idx).cycle
        Next idx
        
        ' remove any net drift from the PEC curve
        drift = (PECCurve(gRAWormPeriod - 1).signal - PECCurve(0).signal) / (gRAWormPeriod + 1)
        CurveMin.signal = 100
        CurveMax.signal = -100
        For idx = 0 To (gRAWormPeriod - 1)
            ' average the signal
            PECCurve(idx).signal = PECCurve(idx).signal - idx * drift
            If PECCurve(idx).signal > CurveMax.signal Then
                CurveMax = PECCurve(idx)
                syncidx = idx
            End If
            If PECCurve(idx).signal < CurveMin.signal Then
                CurveMin = PECCurve(idx)
            End If
        Next idx
        
        seccount = syncidx
        
        MaxPE = CurveMax.signal
        MinPE = CurveMin.signal
        Call PEC_PlotCurve
        
        ' caluculate correction rates to be used.
        Call CalcRates
        
    Else
        HC.Add_Message ("PECImport: Insufficient PEC samples!")
        err = True
    End If
    
    GoTo errCheck
    
ImportError:
    err = True
    HC.Add_Message ("PECImport: Error reading PEC file")
errCheck:
    If err Then
        HC.Command22.Enabled = False
        HC.PECTimer.Enabled = False
        HC.CmdPecSave.Enabled = False
        HC.GainScroll.Enabled = False
        HC.PhaseScroll.Enabled = False
        PEC_Enabled = False
        Import = False
    Else
        HC.PECTimer.Enabled = True
        HC.CmdPecSave.Enabled = True
        HC.GainScroll.Enabled = True
        HC.PhaseScroll.Enabled = True
        HC.Command22.Enabled = True
        Import = True
    End If

End Function
Private Sub Export()

Dim temp1 As String
Dim idx As Integer
Dim pos As Integer

    On Error Resume Next
    
    Open FileDlg.filename For Output As #1
    If err <> 0 Then
      ' File operation(s) failed, handle the error
    Else
'        Print #1, Date$ & " " & Time$
        Print #1, "# " & HC.MainLabel.Caption
        Print #1, "!WormPeriod=" & CStr(gRAWormPeriod)
        Print #1, "# time - motor - smoothed PE"
        For idx = 0 To (gRAWormPeriod - 1)
            pos = (idx + HC.PhaseScroll.Value) Mod gRAWormPeriod
            Print #1, CStr(idx) & " " & CStr(PECCurve(idx).PEPosition) & " " & CStr(PECCurve(pos).signal)
        Next idx
    End If
    Close #1

End Sub


Public Sub PEC_PlotCurve()
Dim idx As Integer
Dim oldval, newval As Double
Dim range As Double
Dim mid As Integer
   
    firsttime = True
    range = MaxPE - MinPE
    mid = HC.plot.ScaleHeight / 2
    HC.plot.Cls
    HC.plot.Line (0, mid)-(HC.plot.ScaleWidth, mid), &H80FF&
    HC.plot.Line (0, (mid) - 4)-(0, mid + 2), &H80FF&
    HC.plot.Line (HC.plot.ScaleWidth * 0.25, mid - 2)-(HC.plot.ScaleWidth * 0.25, mid + 2), &H80FF&
    HC.plot.Line (HC.plot.ScaleWidth * 0.5, mid - 2)-(HC.plot.ScaleWidth * 0.5, mid + 2), &H80FF&
    HC.plot.Line (HC.plot.ScaleWidth * 0.75, (mid) - 2)-(HC.plot.ScaleWidth * 0.75, mid + 2), &H80FF&
    HC.plot.Line (HC.plot.ScaleWidth - 1, mid - 2)-(HC.plot.ScaleWidth - 1, mid + 2), &H80FF&
    
    If range > 0 Then
        oldval = mid - PECCurve(0).signal * 0.8 * HC.plot.ScaleHeight / range
        For idx = 1 To (gRAWormPeriod - 1)
            newval = mid - PECCurve(idx).signal * 0.8 * HC.plot.ScaleHeight / range
            HC.plot.Line (idx * HC.plot.ScaleWidth / gRAWormPeriod, newval)-((idx - 1) * HC.plot.ScaleWidth / gRAWormPeriod, oldval), vbMagenta
            oldval = newval
        Next idx
    End If
End Sub

Private Sub CalcRates()
Dim idx, sec As Integer
Dim ratesum As Double
Dim rate, lastrate, truerate, remainder As Double
Dim stepsmoved As Double
Dim debugfile As String

    ' calculate the rate change between each PE curve sample
    ratesum = 0
    For idx = 1 To (gRAWormPeriod - 1)
        PECCurve(idx).PErate = PECCurve(idx - 1).signal - PECCurve(idx).signal
        ratesum = ratesum + PECCurve(idx).PErate
    Next idx
    ' first rate = average of 2nd and last to remove any discontinuities
    PECCurve(0).PErate = (PECCurve(gRAWormPeriod - 1).PErate + PECCurve(1).PErate) / 2
    ratesum = ratesum + PECCurve(0).PErate
 
    ' Apply the current threshhold and gain settings
    ' The threshold setting allows us to reduce the number of rate corrections sent
    ' to the mount.
    ' The gain is just a user 'fiddle' factor ;-)
    ratesum = 0
    lastrate = PECCurve(gRAWormPeriod - 1).PErate
    For idx = 0 To (gRAWormPeriod - 1)
        rate = PECCurve(idx).PErate
        If Abs(lastrate - rate) > threshold Then
            lastrate = PECCurve(idx).PErate
        Else
            rate = lastrate
        End If
        rate = rate * gain
        PECCurve(idx).PECrate = rate
        ratesum = ratesum + rate
    Next idx
    
    ' The sum of all rate changes over a single cycle should be 0 i.e. sidereal rate
    ' If this isn't the case then adjust them accordingly to ensure there is no net drift
    For idx = 0 To (gRAWormPeriod - 1)
        PECCurve(idx).PECrate = PECCurve(idx).PECrate - (ratesum / gRAWormPeriod)
    Next idx
    
    ' unfortunately the way the mount accepts 'quantised' rate change messages means that we can't have
    ' just any rate we want. So work out what the rate will the mount will actually track at
    ' determine any error and attemp to correct for it in the next sample.
    ' Using this approach we should be able to acheve at worst sidereal - 0.024 arcsecs/sec
    remainder = 0
    ratesum = 0
    For idx = 0 To (gRAWormPeriod - 1)
        If gHemisphere = 0 Then
            rate = SID_RATE_NORTH + PECCurve(idx).PECrate + remainder
            truerate = (9325.46154 / (Int((9325.46154 / rate) + 0.5)))
        Else
            rate = SID_RATE_SOUTH + PECCurve(idx).PECrate + remainder
            truerate = (9325.46154 / (Int((9325.46154 / rate) - 0.5)))
        End If
         
        ' work out the error for next time
        remainder = rate - truerate
        PECCurve(idx).PECrate = truerate
        ratesum = ratesum + (PECCurve(idx).PECrate - gSiderealRate)
    Next idx
    
    HC.Add_Message ("PECImport: RateSum=" & FormatNumber(ratesum, 4))
    
    ' now we now what rates the mount will be tracking at we can
    ' calculate the expected motor positions at each sample
    PECCurve(0).PECPosition = PECCurve(0).PEPosition
    lastrate = 0
    For idx = 1 To (gRAWormPeriod - 1)
        ' motorpos = lastmotorpos + elapsedtime * gTot_RA /  (ARCSECS_PER_360DEGREES / lastRate)
        rate = PECCurve(idx - 1).PECrate
        If rate = 0 Then
            rate = lastrate
        Else
            lastrate = rate
        End If
        
        stepsmoved = gTot_RA / (ARCSECS_PER_360DEGREES / rate)
        newpos = PECCurve(idx - 1).PECPosition + stepsmoved
        PECCurve(idx).PECPosition = newpos Mod (gRAWormSteps)
    Next idx
    
    ' A lot has gone on here so write out a debug file
    ' for anaysis if things don't work as the should.
    debugfile = HC.oPersist.GetIniPath() + "\pec_debug.txt"
    On Error GoTo endsub
    Open debugfile For Output As #1
    Print #1, "Index PE PosRawPE PosPE PosPEC RatePE RatePEC"
    For idx = 0 To (gRAWormPeriod - 1)
        Print #1, CStr(idx) & " " & _
                  FormatNumber(PECCurve(idx).signal, 4) & " " & _
                  FormatNumber(PECCurve(idx).RawPosn, 0) & " " & _
                  FormatNumber(PECCurve(idx).PEPosition, 0) & " " & _
                  FormatNumber(PECCurve(idx).PECPosition, 0) & " " & _
                  FormatNumber(PECCurve(idx).PErate, 4) & " " & _
                  FormatNumber(PECCurve(idx).PECrate, 4)
    Next idx
    Close #1
endsub:

End Sub
        
Private Function GetIdx(idx As Integer) As Integer
Dim motorpos As Double
Dim curvepos As Double
Dim i As Integer
    ' Determine an appropriate index into the PEC table that gives the
    ' best match for the current motor position. The best match is the
    ' position that is one step in advance of the motor.
    ' Generally this will just result in an incermenting of the index so
    ' a starting position is suplied to speed up the number of comparisons
    ' required.

    
    gEmulRA = EQ_GetMotorValues(0)
    gLast_time = EQnow_lst_norange()
    gCurrent_time = gLast_time
    gEmulRA_Init = gEmulRA
    
    If PEC_Enabled Then
        ' PEC tacking - sync with PEC calculated motor positions.
        curvepos = PECCurve(idx).PECPosition
    Else
        ' sidereal track so use uncorrected positions
        curvepos = PECCurve(idx).PEPosition
    End If
    
    i = 0
    If gHemisphere = 0 Then
        ' Northern hemisphere
        ' For northern hemisphere curves the motor positions increase
        ' with increasing index
       motorpos = NormalisePosition(gEmulRA + PhaseShift)
        If (motorpos > curvepos) Then
            While motorpos > curvepos And i < gRAWormPeriod
                ' search forwards till we find a curve position that is
                ' greater than the motor position
                idx = (idx + 1) Mod gRAWormPeriod
                curvepos = PECCurve(idx).PECPosition
                i = i + 1
            Wend
        Else
            While motorpos < curvepos And i < gRAWormPeriod
                ' search backwards till we find a curve position that is
                ' less than the motor position
                idx = idx - 1
                If idx < 0 Then idx = (gRAWormPeriod - 1)
                curvepos = PECCurve(idx).PECPosition
                i = i + 1
            Wend
            ' now increment to next curve position
            ' its best to have the curve in advance of the motor!
            idx = (idx + 1) Mod gRAWormPeriod
        End If
    Else
        ' Southern hemisphere
        ' For southern hemisphere curves the motor positions decrease
        ' with increasing index
        motorpos = NormalisePosition(gEmulRA - PhaseShift)
        If (motorpos > curvepos) Then
            While motorpos > curvepos And i < gRAWormPeriod
                ' search backwards till we find a curve position that is
                ' smaller than the motor position
                idx = idx - 1
                If idx < 0 Then idx = (gRAWormPeriod - 1)
                curvepos = PECCurve(idx).PECPosition
                i = i + 1
            Wend
            ' now increment to next curve position
            ' its best to have the curve in advance of the motor!
            idx = (idx + 1) Mod gRAWormPeriod
        Else
            ' search forwards till we find a curve position that is
            ' greater than the motor position
            While motorpos < curvepos And i < gRAWormPeriod
                idx = (idx + 1) Mod gRAWormPeriod
                curvepos = PECCurve(idx).PECPosition
                i = i + 1
            Wend
        End If
    End If
    
    GetIdx = idx
End Function
